<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreDoctorRequest;
use App\Models\Doctor;
use App\Traits\HttpResponses;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
/**
 * @group Doctor management
 *
 * APIs to manage Doctor resources
 * */
class DoctorController extends Controller
{

    use HttpResponses;
    /**
     * Display all Doctors
     *@response 200 scenario="request was successful"{
     * "status": "Request was successful",
     * "message": "success",
     * "data": [
     * {
     * "id": 8,
     * "name": "mahmoud mohamed",
     * "code": "215615651235",
     * "type": "advisor",
     * "specialization": "surgeon",
     * "work_days": "[\"sunday\",\"monday\",\"friday\"]",
     * "start_time": "12:15:00",
     * "end_time": "13:15:00",
     * "created_at": "2024-10-11T08:07:43.000000Z",
     * "updated_at": "2024-10-11T08:07:43.000000Z"
     * }
     * ]
     * }
     */
    public function index(): JsonResponse
    {
        $doctor = Doctor::all();
//        return response()->json($doctor);
        return $this->success($doctor,"success");
    }


    /*
     * @apiResourceCollection App\Http\Resources\PatientCollection
     * @apiResourceModel App\Models\Patient
     */
    /**
     * create a doctor
     *@response 200 scenario="request was successful"{
     * "status": "Request was successful",
     * "message": "success",
     * "data": {
     * "name": "mahmoud mohamed",
     * "code": "215615651235",
     * "type": "advisor",
     * "specialization": "surgeon",
     * "start_time": "12:15",
     * "end_time": "13:15",
     * "work_days": "[\"sunday\",\"monday\",\"friday\"]",
     * "updated_at": "2024-10-11T08:07:43.000000Z",
     * "created_at": "2024-10-11T08:07:43.000000Z",
     * "id": 8
     * }
     * }
     * @response 422 scenario="missing parameters"{
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "name": [
     * "The name field is required."
     * ],
     * "code": [
     * "The code field is required."
     * ],
     * "type": [
     * "The type field is required."
     * ],
     * "specialization": [
     * "The specialization field is required."
     * ],
     * "work_days": [
     * "The work_days field is required."
     * ],
     * "start_time": [
     * "The start time field is required."
     * ],
     * "end_time": [
     * "The end time field is required."
     * ]
     * }
     * }
     * @response 404 scenario="name or code already exists"{
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "name": [
     * "This doctor name already exists."
     * ],
     * "code": [
     * "This doctor code already exists."
     * ]
     * }
     * }
     *
     * @bodyParam name string required The doctor's full name. Example: Dr. John Doe
     * @bodyParam code string required A unique doctor code. Example: DOC123
     * @bodyParam type string required The type of the doctor. Can be 'General' or 'Specialist'. Example: General
     * @bodyParam specialization string required The doctor's specialization. Example: Cardiologist
     * @bodyParam work_days string[] required The working days for the doctor. You can enter between 1 and 7 values. Valid values are 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'. Example: ["Monday", "Wednesday", "Friday"]
     * @bodyParam start_time string required The doctor's start time in 'HH:MM' format. Must be earlier than the end time. Example: 09:00
     * @bodyParam end_time string required The doctor's end time in 'HH:MM' format. Must be later than the start time. Example: 17:00
     *
     */
    public function store(StoreDoctorRequest $request): JsonResponse
    {
        $validatedData=$request->validated();
        $validatedData['work_days'] = json_encode($validatedData['work_days'],true);
//        $validatedData['work_days'] =  array_values(json_decode($validatedData['work_days'], true));
        $doctor = Doctor::create($validatedData);
        return $this->success($doctor,"success");
    }


    /**
     * Display doctor by id
     * @param $id
     * @return JsonResponse
     * @response 404 scenario="There is no doctor with this ID""{
     * "status": "error",
     * "message": "There is no doctor with this ID",
     * "data": ""
     * }
     * @response 200 scenario="request was successful"
     * {
     * "status": "Request was successful",
     * "message": "success",
     * "data": {
     * "id": 1,
     * "name": "mahmoud ali",
     * "code": "215615651231",
     * "type": "advisor",
     * "specialization": "surgeon",
     * "work_days": "[\"sunday\",\"monday\",\"friday\"]",
     * "start_time": "12:15:00",
     * "end_time": "13:15:00",
     * "created_at": "2024-09-30T05:22:36.000000Z",
     * "updated_at": "2024-09-30T05:22:36.000000Z"
     * }
     * }
 */
    public function show($id): JsonResponse
    {
        try {
        $doctor = Doctor::findOrFail($id);
//        return response()->json($doctor);
        return $this->success($doctor,"success");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no doctor with this ID", 404);
        }
    }

    /**
     * Update a doctor
     * @apiResourceCollection App\Http\Resources\DoctorCollection
     * @apiResourceModel App\Models\Doctor
     *
     * @bodyParam name string required The doctor's full name. Example: Dr. John Doe
     * @bodyParam code string required A unique doctor code. Example: DOC123
     * @bodyParam type string required The type of the doctor. Can be 'General' or 'Specialist'. Example: General
     * @bodyParam specialization string required The doctor's specialization. Example: Cardiologist
     * @bodyParam work_days string[] required The working days for the doctor. You can enter between 1 and 7 values. Valid values are 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'. Example: ["Monday", "Wednesday", "Friday"]
     * @bodyParam start_time string required The doctor's start time in 'HH:MM' format. Must be earlier than the end time. Example: 09:00
     * @bodyParam end_time string required The doctor's end time in 'HH:MM' format. Must be later than the start time. Example: 17:00
     *
     */
    public function update(StoreDoctorRequest $request, $id):JsonResponse
    {



        try {
        $doctor = Doctor::findOrFail($id);
        $validatedData = $request->validated();
        $validatedData['work_days'] = json_encode($validatedData['work_days'],true);
        $doctor->update($validatedData);
        return $this->success($validatedData,"the doctor has been updated successfully");

        } catch (ModelNotFoundException $e) {
            // If no service is found, return a 404 error response
            return $this->error("", "There is no doctor with this ID", 404);
        }
    }
    /**
     * Delete Doctor
     * @response 200 scenario="the doctor has been deleted successfully"
     * {
     * "status": "Request was successful",
     * "message": "the doctor has been deleted successfully",
     * "data": ""
     * }
     *
     * @response 404 scenario="there is no doctor with the specified id"
     * {
     * "status": "error",
     * "message": "There is no Doctor with this ID",
     * "data": ""
     * }
     *
     *
     */
    public function destroy($id): JsonResponse
    {

        try {
        $doctor = Doctor::findOrFail($id);
        $doctor->delete();
        return $this->success("","the doctor has been deleted successfully");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no Doctor with this ID", 404);
        }
    }
}
