<?php

namespace App\Http\Controllers;

use App\Http\Requests\StorePatientAttachmentRequest;
use App\Http\Resources\PatientAttachmentCollection;
use App\Http\Resources\PatientAttachmentResource;
use App\Http\Resources\PatientResource;
use App\Models\Patient;
use App\Models\PatientAttachment;
use App\Traits\HttpResponses;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;
/**
 * @group Patient attachment management
 *
 * APIs to manage Patient attachment resources
 * */
class PatientAttachmentController extends Controller
{
    use HttpResponses;
    /**
     * Display all patients attachments
     * @return JsonResponse
     * @response 200 scenario="request was successful"
     * {
     * "status": "Request was successful",
     * "message": "Patient attachments retrieved successfully",
     * "data": [
     * {
     * "id": 1,
     * "patient_id": 4,
     * "civil_id_doc": "ali_mohamed/ali_mohamed_civil_id_doc.jpeg",
     * "patient_status_doc": "ali_mohamed/ali_mohamed_patient_status_doc.jpg",
     * "report_1_doc": "ali_mohamed/ali_mohamed_report_1_doc.jpeg",
     * "report_2_doc": "ali_mohamed/ali_mohamed_report_2_doc.jpg",
     * "other_document_doc": "ali_mohamed/ali_mohamed_other_document_doc.jpeg"
     * },
     * {
     * "id": 2,
     * "patient_id": 5,
     * "civil_id_doc": "ali_saeed/ali_saeed_civil_id_doc.jpeg",
     * "patient_status_doc": "ali_saeed/ali_saeed_patient_status_doc.jpg",
     * "report_1_doc": "ali_saeed/ali_saeed_report_1_doc.jpeg",
     * "report_2_doc": "ali_saeed/ali_saeed_report_2_doc.jpg",
     * "other_document_doc": "ali_mohamed/ali_mohamed_other_document_doc.jpeg"
     * }
     * ]
     * }
     *
     */
    public function index(): JsonResponse
    {
        $patientAttachments = PatientAttachment::all();
        return $this->success(new PatientAttachmentCollection($patientAttachments), "Patient attachments retrieved successfully");
    }



//    public function storeAttachment(StorePatientAttachmentRequest $request): JsonResponse
//    {
//        $validatedData = $request->only(['civil_id', 'civil_id_doc', 'patient_status_doc', 'report_1_doc', 'report_2_doc', 'other_document_doc']);
//        $patient = Patient::where('civil_id', $validatedData['civil_id'])->firstOrFail();
//        $patient_name = $patient->name_english; // Example: 'Ahmed Ali'
//
//        $patient_name = str_replace(' ', '_', trim($patient_name));
//
//        try {
//            $paths = [];
//
//            // Check and store each file
//            if (isset($validatedData['civil_id_doc'])) {
//                $civil_id_doc_name = $patient_name . '_civil_id_doc.' . $validatedData['civil_id_doc']->getClientOriginalExtension();
//                $paths['civil_id_doc'] = $validatedData['civil_id_doc']->storeAs('patientDocs', $civil_id_doc_name, 'public');
//            }
//
//            if (isset($validatedData['patient_status_doc'])) {
//                $patient_status_doc_name = $patient_name . '_patient_status_doc.' . $validatedData['patient_status_doc']->getClientOriginalExtension();
//                $paths['patient_status_doc'] = $validatedData['patient_status_doc']->storeAs('patientDocs', $patient_status_doc_name, 'public');
//            }
//
//            if (isset($validatedData['report_1_doc'])) {
//                $report_1_doc_name = $patient_name . '_report_1_doc.' . $validatedData['report_1_doc']->getClientOriginalExtension();
//                $paths['report_1_doc'] = $validatedData['report_1_doc']->storeAs('patientDocs', $report_1_doc_name, 'public');
//            }
//
//            if (isset($validatedData['report_2_doc'])) {
//                $report_2_doc_name = $patient_name . '_report_2_doc.' . $validatedData['report_2_doc']->getClientOriginalExtension();
//                $paths['report_2_doc'] = $validatedData['report_2_doc']->storeAs('patientDocs', $report_2_doc_name, 'public');
//            }
//
//            if (isset($validatedData['other_document_doc'])) {
//                $other_document_doc_name = $patient_name . '_other_document_doc.' . $validatedData['other_document_doc']->getClientOriginalExtension();
//                $paths['other_document_doc'] = $validatedData['other_document_doc']->storeAs('patientDocs', $other_document_doc_name, 'public');
//            }
//
//            // Update the patient's documents with the file paths
//            $patient->update($paths);
//
//            return $this->success([$patient], "attachment created successfully",);
//
//        } catch (\Exception $e) {
//            // Handle exceptions, e.g., if the patient is not found or if there's a validation error
//            return response()->json(['error' => $e->getMessage()], 500);
//        }
//    }



    /*
     * Store a newly created resource in storage.
     * @apiResourceCollection App\Http\Resources\PatientAttachmentCollection
     * @apiResourceModel App\Models\PatientAttachment
     */
    /**
     * Create new patient attachment
     * @param StorePatientAttachmentRequest $request
     * @return JsonResponse
     * @response 200 scenario="patient attachment has been created successflly"
     *  {
     *  "status": "success",
     *  "data": {
     *  "patient_id": "4",
     *  "civil_id_doc": "ali_mohamed/ali_mohamed_civil_id_doc.jpeg",
     *  "patient_status_doc": "ali_mohamed/ali_mohamed_patient_status_doc.jpg",
     *  "report_1_doc": "ali_mohamed/ali_mohamed_report_1_doc.jpeg",
     *  "report_2_doc": "ali_mohamed/ali_mohamed_report_2_doc.jpg",
     *  "other_document_doc": "ali_mohamed/ali_mohamed_other_document_doc.jpeg",
     *  "updated_at": "2024-10-12T15:17:23.000000Z",
     *  "created_at": "2024-10-12T15:17:23.000000Z",
     *  "id": 1
     *  }
     *  }
     * @response 404 scenario="there is no patient for this id to add attachment to"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "patient_id": [
     * "The selected patient id is invalid."
     * ]
     * }
     * }
     * @response 404 scenario="the patient already has attachments assigned to"
     * {
     * "status": "error",
     * "message": "this patient already has attachment assigned to it",
     * "data": {}
     * }
     */
    public function store(StorePatientAttachmentRequest $request): JsonResponse
    {
        $validatedData = $request->all();
        try {//search if there are previously assigned attachment for that user
            $patientAttachment = PatientAttachment::where('patient_id',$validatedData['patient_id']);
        }catch (ModelNotFoundException $e){

            // Use findOrFail to ensure the patient exists or throw a 404 error
            $patient = Patient::findOrFail($validatedData['patient_id']);//will check if the patient id is valid
            $patientName = Str::replace(' ', '_', trim($patient->name_english));

            $documents = [
                'civil_id_doc' => '_civil_id_doc',
                'patient_status_doc' => '_patient_status_doc',
                'report_1_doc' => '_report_1_doc',
                'report_2_doc' => '_report_2_doc',
                'other_document_doc' => '_other_document_doc',
            ];

            $paths = [];

            foreach ($documents as $key => $suffix) {
                if (isset($validatedData[$key])) {
                    $fileName = $patientName . $suffix . '.' . $validatedData[$key]->getClientOriginalExtension();
                    $paths[$key] = $validatedData[$key]->storeAs($patientName, $fileName, 'public');
                } else {
                    $paths[$key] = null; // Handle case where a document isn't uploaded
                }
            }

            try {
                $patientAttachments = PatientAttachment::create([
                    'patient_id' => $validatedData['patient_id'],
                    'civil_id_doc' => $paths['civil_id_doc'],
                    'patient_status_doc' => $paths['patient_status_doc'],
                    'report_1_doc' => $paths['report_1_doc'],
                    'report_2_doc' => $paths['report_2_doc'],
                    'other_document_doc' => $paths['other_document_doc'],
                ]);

                return response()->json([
                    'status' => 'success',
                    'data' => $patientAttachments
                ]);

            } catch (Exception $e) {
                return response()->json(['error' => $e->getMessage()], 500);
            }

        }
    return $this->error($patientAttachment,"this patient already has attachment assigned to it",404);

    }



    /**
     * Display Patient attachments by id
     * @param $id
     * @return JsonResponse
     *
     * @response 200 scenario="success"
     * {
     * "status": "Request was successful",
     * "message": "patient attachments returned successfully",
     * "data": {
     * "id": 2,
     * "patient_id": 4,
     * "civil_id_doc": "ali_mohamed/ali_mohamed_civil_id_doc.jpeg",
     * "patient_status_doc": "ali_mohamed/ali_mohamed_patient_status_doc.jpg",
     * "report_1_doc": "ali_mohamed/ali_mohamed_report_1_doc.jpeg",
     * "report_2_doc": "ali_mohamed/ali_mohamed_report_2_doc.jpg",
     * "other_document_doc": "ali_mohamed/ali_mohamed_other_document_doc.jpeg"
     * }
     * }
     *
     * @response 404 scenario="no patient attachments found with this id"
     * {
     * "status": "error",
     * "message": "There is no patient attachment with this ID",
     * "data": ""
     * }
     */
    public function show($id):JsonResponse
    {//show the attachments for a specific user
        try {
        $patientAttachments=PatientAttachment::findOrFail($id);
        return $this->success(new PatientAttachmentResource($patientAttachments),"patient attachments returned successfully");

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no patient attachment with this ID", 404);
        }

    }


    /**
     *  Update patient attachments
     * @param StorePatientAttachmentRequest $request
     * @param $id
     * @return JsonResponse
     * @response 200 scenario="patient attachment has been updated successfully"
     * {
     * "status": "Request was successful",
     * "message": "patient attachment has been updated successfully",
     * "data": {
     * "id": 1,
     * "patient_id": "4",
     * "civil_id_doc": "ali_mohamed/ali_mohamed_civil_id_doc.jpeg",
     * "patient_status_doc": "ali_mohamed/ali_mohamed_patient_status_doc.jpg",
     * "report_1_doc": "ali_mohamed/ali_mohamed_report_1_doc.jpeg",
     * "report_2_doc": "ali_mohamed/ali_mohamed_report_2_doc.jpg",
     * "other_document_doc": "ali_mohamed/ali_mohamed_other_document_doc.jpeg"
     * }
     * }
     * @response 404 scenario="no patient attachments found with this id"
     * {
     * "status": "error",
     * "message": "There is no patient attachment with this ID",
     * "data": ""
     * }
     */
    public function update(StorePatientAttachmentRequest $request, $id): JsonResponse
    {


        try {

            $validatedData = $request->all();
            $patientAttachment = PatientAttachment::findOrFail($id);
            $patient = Patient::findOrFail($validatedData['patient_id']);
            $patientName = Str::replace(' ', '_', trim($patient->name_english));

            $documents = [
                'civil_id_doc' => '_civil_id_doc',
                'patient_status_doc' => '_patient_status_doc',
                'report_1_doc' => '_report_1_doc',
                'report_2_doc' => '_report_2_doc',
                'other_document_doc' => '_other_document_doc',
            ];

            $paths = [];

            foreach ($documents as $key => $suffix) {//this will extract file extension ,setup file names and store in storage/public/$patientName/filename
                if (isset($validatedData[$key])) {
                    $fileName = $patientName . $suffix . '.' . $validatedData[$key]->getClientOriginalExtension();
                    $paths[$key] = $validatedData[$key]->storeAs($patientName, $fileName, 'public');
                } else {
                    $paths[$key] = $patientAttachment->$key; // Retain the existing path if no new file is uploaded
                }
            }

            try {
                $patientAttachment->update([
                    'patient_id' => $validatedData['patient_id'],
                    'civil_id_doc' => $paths['civil_id_doc'],
                    'patient_status_doc' => $paths['patient_status_doc'],
                    'report_1_doc' => $paths['report_1_doc'],
                    'report_2_doc' => $paths['report_2_doc'],
                    'other_document_doc' => $paths['other_document_doc'],
                ]);
                return $this->success(new PatientAttachmentResource($patientAttachment), "patient attachment was updated successfully");

//                return response()->json([
//                    'status' => 'success',
//                    'data' => $patientAttachment,
//                ]);

            } catch (Exception $e) {
                return response()->json(['error' => $e->getMessage()], 500);
            }

        } catch (ModelNotFoundException $e) {
            // If no service is found, return a 404 error response
            return $this->error("", "There is no patient attachment with this ID", 404);
        }


    }


    /**
     * Delete patient attachment
     * @param $id
     * @return JsonResponse
     * @response 200 scenario="patient attachments deleted successfully"
     *{
     * "status": "Request was successful",
     * "message": "the patient attachments  has been deleted successfully",
     * "data": null
     * }
     *
     * @response 404 scenario="is no patient attachment with the specified ID"
     * {
     * "status": "error",
     * "message": "There is no patient attachment with this ID",
     * "data": ""
     * }
     */
    public function destroy($id): JsonResponse
    {//delete the attachments for a specific user

        try {
        $patientAttachment = PatientAttachment::findOrFail($id);
        //$patientAttachment->destroy();
        $patientAttachment->delete();
        return $this->success(null,"the patient attachments  has been deleted successfully");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no patient attachment with this ID", 404);
        }
    }
}
