<?php

namespace App\Http\Controllers;

use App\Http\Requests\StorePatientRequest;
use App\Http\Resources\PatientCollection;
use App\Http\Resources\PatientResource;
use App\Models\Patient;
use App\Traits\HttpResponses;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use function PHPUnit\Framework\returnArgument;

/**
 * @group Patient management
 *
 * APIs to manage Patient resources
 * */
class PatientController extends Controller
{
    use HttpResponses;
    /**
     * Display all patients
     * @return JsonResponse
     * @response 200 scenario="success"
     * {
     * "status": "Request was successful",
     * "message": "success",
     * "data": [
     * {
     * "id": 2,
     * "name_english": "ali mohamed",
     * "name_arabic": "أحمد علي",
     * "email": "ahmed@gmail.com",
     * "civil_id": "30201078908135",
     * "phone_number": "01145684288",
     * "phone_home": "01145684288",
     * "nationality": "egyption",
     * "city": "menofia",
     * "gender": "male",
     * "birth_date": "2002-08-08",
     * "address": "64 altowheed stread",
     * "insurance_company": "bank masr for life ensurance",
     * "policy_holder": "mohomed hatem",
     * "policy_number": "2262509",
     * "policy_start_date": "2006-08-08",
     * "policy_end_date": "2025-08-09",
     * "notice": "i noticed",
     * "referral": "internet",
     * "patient_status": null,
     * "report_1": null,
     * "report_2": null,
     * "other_document": null,
     * "transactions": []
     * }
     * ]
     * }
     */
    public function index(): JsonResponse
    {
        $patient = Patient::all();
        return $this->success(new PatientCollection($patient),"success");
    }


    /*
     * store new patient
     * @bodyParam name_english string required The patient's name in English. Example: ahmed ismail
     * @bodyParam name_arabic string required The patient's name in Arabic. Example: أحمد إسماعيل
     * @bodyParam civil_id string required The patient's 14-digit civil ID. Example: 30303089752169
     * @bodyParam phone_number string required The patient's phone number. Example: 01025984523
     * @bodyParam nationality string required The patient's nationality. Example: egyption
     * @bodyParam city string required The city where the patient lives. Example: menofia
     * @bodyParam gender string required The patient's gender. Must be 'male' or 'female'. Example: male
     * @bodyParam birth_date date required The patient's birth_date in 'YYYY-MM-DD' format. Example: 2000-03-20
     * @bodyParam address string optional The patient's address. Example: 15-talaat harb street
     * @bodyParam insurance_company string optional The insurance company name. Example: insurance company
     * @bodyParam policy_holder string optional The policy_holder name. Example: policy holder
     * @bodyParam policy_number string optional The policy number. Example: 025491502
     * @bodyParam policy_start_date date optional The policy start date in 'YYYY-MM-DD' format. Example: 2003-03-03
     * @bodyParam policy_end_date date optional The policy end date in 'YYYY-MM-DD' format. Example: 2024-04-04
     */

     /**
      *Create patient
      * @param StorePatientRequest $request
      * @return JsonResponse
      * @response 200 scenario="patient has been created successfully"
      * {
      * "status": "Request was successful",
      * "message": "patient has been created successfully",
      * "data": {
      * "id": 2,
      * "name_english": "ali mohamed",
      * "name_arabic": "أحمد علي",
      * "email": "ahmed@gmail.com",
      * "civil_id": "30201078908135",
      * "phone_number": "01145684288",
      * "phone_home": "01145684288",
      * "nationality": "egyption",
      * "city": "menofia",
      * "gender": "male",
      * "birth_date": "2002-8-8",
      * "address": "64 altowheed stread",
      * "insurance_company": "bank masr for life ensurance",
      * "policy_holder": "mohomed hatem",
      * "policy_number": "2262509",
      * "policy_start_date": "2006-8-8",
      * "policy_end_date": "2025-8-9",
      * "notice": "i noticed",
      * "referral": "internet",
      * "patient_status": null,
      * "report_1": null,
      * "report_2": null,
      * "other_document": null,
      * "transactions": []
      * }
      * }
      * @response 404 scenario="civil id or email already exist"
      * {
      * "status": "error",
      * "message": "Validation failed",
      * "data": {
      * "civil_id": [
      * "This civil id already exists."
      * ],
      * "email": [
      * "The email has already been taken."
      * ]
      * }
      * }
      *
      *
      *
      * @bodyParam name_english string required The patient's name in English. Example: ahmed ismail
      * @bodyParam name_arabic string optional The patient's name in Arabic. Example: أحمد إسماعيل
      * @bodyParam email email required The patient's email. Example: ahmed@gmail.com
      * @bodyParam civil_id string required The patient's 14-digit civil ID. Example: 30303089752169
      * @bodyParam phone_number string optional The patient's phone number. Example: 01025984523
      * @bodyParam phone_home string optional The patient's phone number. Example: 01025984523
      * @bodyParam nationality string required The patient's nationality. Example: egyption
      * @bodyParam city string optional The city where the patient lives. Example: menofia
      * @bodyParam gender string required The patient's gender. Must be 'male' or 'female'. Example: male
      * @bodyParam birth_date date required The patient's birth_date in 'YYYY-MM-DD' format. Example: 2000-03-20
      * @bodyParam address string optional The patient's address. Example: 15-talaat harb street
      * @bodyParam insurance_company string optional The insurance company name. Example: insurance company
      * @bodyParam policy_holder string optional The policy_holder name. Example: policy holder
      * @bodyParam policy_number string optional The policy number. Example: 025491502
      * @bodyParam policy_start_date date optional The policy start date in 'YYYY-MM-DD' format. Example: 2003-03-03
      * @bodyParam policy_end_date date optional The policy end date in 'YYYY-MM-DD' format. Example: 2024-04-04
      */
    public function store(StorePatientRequest $request): JsonResponse
    {
        $validatedData = $request->validated();
        $patient = Patient::create($validatedData);
//        return $this->success(
//            ['patient'=> $patient]
//        );
        return $this->success(new PatientResource($patient),"patient has been created successfully");
    }
    
    
        public function showById($id): JsonResponse
    {
        $patient = Patient::findOrFail($id);
//        return response()->json($patient);
        return $this->success(new PatientResource($patient),"Patient has been returned successfully");
    }


    /**
     * Display patient by id
     * @param $id
     * @return JsonResponse
     * @response 200 scenario="request was successful"
     * {
     * "status": "Request was successful",
     * "message": "success",
     * "data": {
     * "id": 2,
     * "name_english": "ali mohamed",
     * "name_arabic": "أحمد علي",
     * "email": "ahmed@gmail.com",
     * "civil_id": "30201078908135",
     * "phone_number": "01145684288",
     * "phone_home": "01145684288",
     * "nationality": "egyption",
     * "city": "menofia",
     * "gender": "male",
     * "birth_date": "2002-08-08",
     * "address": "64 altowheed stread",
     * "insurance_company": "bank masr for life ensurance",
     * "policy_holder": "mohomed hatem",
     * "policy_number": "2262509",
     * "policy_start_date": "2006-08-08",
     * "policy_end_date": "2025-08-09",
     * "notice": "i noticed",
     * "referral": "internet",
     * "patient_status": null,
     * "report_1": null,
     * "report_2": null,
     * "other_document": null,
     * "transactions": []
     * }
     * }
     * @response 404 scenario="no patient with the specified ID"
     * {
     * "status": "error",
     * "message": "no patient been found",
     * "data": ""
     * }
     */
   public function show(Request $request): JsonResponse
    {
//        dd('sdlkjlk');
        $searchTerm = $request->input('search');
        $option = $request->input('option');

        if (empty($searchTerm)) {
            return $this->error("", "Search term is required", 400);
        }

        if (empty($option)) {
            return $this->error("", "Option is required", 400);
        }

        $validOptions = ['all', 'name', 'file_number', 'civil_id', 'phone'];
        if (!in_array($option, $validOptions)) {
            return $this->error("", "Invalid option provided", 400);
        }

        try {
            $query = Patient::query();

            if ($option == 'all') {
                $query->where(function ($q) use ($searchTerm) {
                    $q->where('id', $searchTerm)
                        ->orWhere('name_english', 'LIKE', '%' . $searchTerm . '%')
                        ->orWhere('name_arabic', 'LIKE', '%' . $searchTerm . '%')
                        ->orWhere('civil_id', 'LIKE', '%' . $searchTerm . '%')
                        ->orWhere('phone_number', 'LIKE', '%' . $searchTerm . '%')
                        ->orWhere('phone_home', 'LIKE', '%' . $searchTerm . '%');
                });
            } elseif ($option == 'file_number') {
                $query->where('id', $searchTerm);
            } elseif ($option == 'name') {
                $query->where('name_english', 'LIKE', '%' . $searchTerm . '%')
                    ->orWhere('name_arabic', 'LIKE', '%' . $searchTerm . '%');
            } elseif ($option == 'civil_id') {
                $query->where('civil_id', 'LIKE', '%' . $searchTerm . '%');
            } elseif ($option == 'phone') {
                $query->where('phone_number', 'LIKE', '%' . $searchTerm . '%')
                    ->orWhere('phone_home', 'LIKE', '%' . $searchTerm . '%');
            }

            $patients = $query->get();

            if ($patients->isEmpty()) {
                return $this->error("", "No patients found matching the criteria", 404);
            }

            return $this->success(PatientResource::collection($patients), "Patients retrieved successfully");

        } catch (\Exception $e) {
            \Log::error('Patient search error: ' . $e->getMessage());
            return $this->error("", "An error occurred during search", 500);
        }
    }
    // public function show(Request $request): JsonResponse
    // {
    //     $searchTerm = $request->input('search'); // Get the search term from the request

    //     if (empty($searchTerm)) {
    //         return $this->error("", "Search term is required", 400);
    //     }

    //     try {
    //         $query = Patient::query();

    //         // Check if search term is numeric (for ID or phone number)
    //         if (is_numeric($searchTerm)) {
    //             $query->where(function ($q) use ($searchTerm) {
    //                 $q->where('id', $searchTerm)
    //                     ->orWhere('phone_number', 'LIKE', '%' . $searchTerm . '%')
    //                     ->orWhere('phone_home', 'LIKE', '%' . $searchTerm . '%')
    //                     ->orWhere('civil_id', 'LIKE', '%' . $searchTerm . '%');
    //             });
    //         } else {
    //             // Non-numeric search across text fields
    //             $query->where(function ($q) use ($searchTerm) {
    //                 $q->where('name_english', 'LIKE', '%' . $searchTerm . '%')
    //                     ->orWhere('name_arabic', 'LIKE', '%' . $searchTerm . '%')
    //                     ->orWhere('insurance_company', 'LIKE', '%' . $searchTerm . '%')
    //                     ->orWhere('gender', 'LIKE', '%' . $searchTerm . '%')
    //                     ->orWhere('city', 'LIKE', '%' . $searchTerm . '%');
    //             });
    //         }

    //         $patients = $query->get();

    //         // Check if patients are found
    //         if ($patients->isEmpty()) {
    //             return $this->error("", "No patients found", 404);
    //         }

    //         return $this->success(PatientResource::collection($patients), "Patients retrieved successfully");

    //     } catch (\Exception $e) {
    //         // Log the error
    //         \Log::error('Patient search error: ' . $e->getMessage());
    //         return $this->error("", "An error occurred during search", 500);
    //     }
    // }

//    public function show(Request $request): JsonResponse
//
//    {
//        $searchTerm = $request->input('search'); // Get the search term from the request
//        try {
//
//            // Use a query to search by id, name, or phone
//
//            $patients = Patient::where('id', $searchTerm)
//                ->orWhere('name_english', 'LIKE', '%' . $searchTerm . '%')
//                ->orWhere('name_arabic', 'LIKE', '%' . $searchTerm . '%')
//                ->orWhere('civil_id', 'LIKE', '%' . $searchTerm . '%')
//                ->orWhere('insurance_company', 'LIKE', '%' . $searchTerm . '%')
//                ->orWhere('gender', 'LIKE', '%' . $searchTerm . '%')
//                ->orWhere('city', 'LIKE', '%' . $searchTerm . '%')
//                ->orWhere('phone_number', 'LIKE', '%' . $searchTerm . '%')
//                ->orWhere('phone_home', 'LIKE', '%' . $searchTerm . '%')
//                ->get(); // This returns a collection of all matching patients
//
//            return $this->success(new PatientCollection($patients), "success");
//
//        } catch (ModelNotFoundException $e) {
//
//            return $this->error("", "no patient been found", 404);
//
//        }
//
//    }
    /**
     * Update patient.
     * @param StorePatientRequest $request
     * @param $id
     * @return JsonResponse
     * @response 404 scenario="no patient with the specified ID"
     *  {
     *  "status": "error",
     *  "message": "There is no patient with this ID",
     *  "data": ""
     *  }
     * @response 200 scenario="patient has been updated successfully"
     * {
     * "status": "Request was successful",
     * "message": "the patient was updated successfully",
     * "data": {
     * "id": 3,
     * "name_english": "abdelfatah mahmoud housein",
     * "name_arabic": "أحمد علي",
     * "email": "ahmed@gmail.com",
     * "civil_id": "30204532890541",
     * "phone_number": "01145684288",
     * "phone_home": "01145684288",
     * "nationality": "egyption",
     * "city": "menofia",
     * "gender": "male",
     * "birth_date": "2002-8-20",
     * "address": "64 altowheed stread",
     * "insurance_company": "bank masr for life ensurance",
     * "policy_holder": "mohomed hatem",
     * "policy_number": "2262509",
     * "policy_start_date": "2006-8-8",
     * "policy_end_date": "2025-8-9",
     * "notice": "i noticed",
     * "referral": "internet",
     * "patient_status": null,
     * "report_1": null,
     * "report_2": null,
     * "other_document": null,
     * "transactions": []
     * }
     * }
     *
     * @bodyParam name_english string required The patient's name in English. Example: ahmed ismail
     * @bodyParam name_arabic string optional The patient's name in Arabic. Example: أحمد إسماعيل
     * @bodyParam email email required The patient's email. Example: ahmed@gmail.com
     * @bodyParam civil_id string required The patient's 14-digit civil ID. Example: 30303089752169
     * @bodyParam phone_number string required The patient's phone number. Example: 01025984523
     * @bodyParam phone_home string optional The patient's phone number. Example: 01025984523
     * @bodyParam nationality string required The patient's nationality. Example: egyption
     * @bodyParam city string optional The city where the patient lives. Example: menofia
     * @bodyParam gender string required The patient's gender. Must be 'male' or 'female'. Example: male
     * @bodyParam birth_date date required The patient's birth_date in 'YYYY-MM-DD' format. Example: 2000-03-20
     * @bodyParam address string optional The patient's address. Example: 15-talaat harb street
     * @bodyParam insurance_company string optional The insurance company name. Example: insurance company
     * @bodyParam policy_holder string optional The policy_holder name. Example: policy holder
     * @bodyParam policy_number string optional The policy number. Example: 025491502
     * @bodyParam policy_start_date date optional The policy start date in 'YYYY-MM-DD' format. Example: 2003-03-03
     * @bodyParam policy_end_date date optional The policy end date in 'YYYY-MM-DD' format. Example: 2024-04-04
     */
    public function update(StorePatientRequest $request, $id):JsonResponse
    {
        try {
            $patient = Patient::findOrFail($id);
            $validatedData = $request->validated();
            $patient->update($validatedData);
            return $this->success(new PatientResource($patient), "the patient was updated successfully");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no patient with this ID", 404);
        }
    }
    /**
     * Delete patient.
     * @param $id
     * @return JsonResponse
     * @response 404 scenario="no patient with the specified ID"
     *   {
     *   "status": "error",
     *   "message": "There is no patient with this ID",
     *   "data": ""
     *   }
     *
     * @response 200 scenario="patient has been deleted successfully"
     * {
     * "status": "Request was successful",
     * "message": "the patient has been  deleted successfully",
     * "data": null
     * }
     */
    public function destroy($id): JsonResponse
    {
        try {
            $patient = Patient::findOrFail($id);
            $patient->delete();
            return $this->success(null,"the patient has been  deleted successfully");

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no patient with this ID", 404);
        }
    }
}
