<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreServiceGroupRequest;
use App\Http\Resources\ServiceGroupCollection;
use App\Http\Resources\ServiceGroupResource;
use App\Models\Service;
use App\Models\ServiceGroup;
use App\Traits\HttpResponses;
use http\Env\Response;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
/**
 * @group Service group management
 *
 * APIs to manage service group resources
 * */
class ServiceGroupController extends Controller
{
    use HttpResponses;
    /**
     * Display all service groups.
     * @return JsonResponse
     */
    public function index(): JsonResponse
    {
        $serviceGroup = ServiceGroup::all();
//        return response()->json($serviceGroup);
        return $this->success(new ServiceGroupCollection($serviceGroup),"service groups returned successfully");
    }
    /**
     * Display service group by id.
     * @param $id
     * @return JsonResponse
     * @response 200 scenario="the service group returned successfully"
     *{
     * "status": "Request was successful",
     * "message": "service group returned successfully",
     * "data": {
     * "id": 7,
     * "code": "4512",
     * "name": "teeth"
     * }
     * }
     * @response 404 scenario="no service group with the provided id"
     * {
     * "status": "error",
     * "message": "There is no service group with this ID",
     * "data": ""
     * }
     */
    public function show($id): JsonResponse
    {
        try {
            $serviceGroup = ServiceGroup::findOrFail($id);
            return $this->success(new ServiceGroupResource($serviceGroup),"service group returned successfully");

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service group with this ID", 404);
        }
    }

//     * @apiResourceCollection App\Http\Resources\ServiceGroupCollection
    /**
     * Create Service Group
     * @apiResourceModel App\Models\ServiceGroup
     * @param StoreServiceGroupRequest $request
     * @return JsonResponse
     * @response 200 scenario="the service group created successfully"
     * {
     * "status": "Request was successful",
     * "message": "the service group has been created successfully",
     * "data": {
     * "id": 8,
     * "code": "45122",
     * "name": "teeth"
     * }
     * }
     * }
     * @response 422 scenario="required parameters missing"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "code": [
     * "the service group code is required"
     * ],
     * "name": [
     * "service group name is required"
     * ]
     * }
     * }
     * @response 404 scenario="code or name already exist"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "code": [
     * "the specified code already exists"
     * ],
     * "name": [
     * "the specified name already exists"
     * ]
     * }
     * }
     *
     */
    public function store(StoreServiceGroupRequest $request):JsonResponse
    {
        $validatedData=$request->validated();
        $serviceGroup=ServiceGroup::create($validatedData);
        return $this->success(new ServiceGroupResource($serviceGroup),"the service group has been created successfully");
    }

    /**
     * Update service group
     * @param StoreServiceGroupRequest $request
     * @param $id
     * @return JsonResponse
     * @response 200 scenario="the service group has been update successfully"
     * {
     * "status": "Request was successful",
     * "message": "the service group has been updated successfully",
     * "data": {
     * "id": 7,
     * "code": "4515616",
     * "name": "surgery"
     * }
     * }
     * }
     * @response 404 scenario="code or name already exist"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "code": [
     * "the specified code already exists"
     * ],
     * "name": [
     * "the specified name already exists"
     * ]
     * }
     * }
     * @response 404 scenario="There is no service group with the specified ID"
     * {
     * "status": "error",
     * "message": "There is no service group with this ID",
     * "data": ""
     * }
     */
    public function update(StoreServiceGroupRequest $request, $id): JsonResponse
    {
        try {
            $serviceGroup = ServiceGroup::findOrFail($id)->first();
//            dd($serviceGroup);
            $validatedData = $request->validated();
//            dd($validatedData);
            $serviceGroup->update($validatedData);
//            dd('hello');
            return $this->success(new ServiceGroupResource($serviceGroup), "the service group has been updated successfully");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service group with this ID", 404);
        }
    }

    /**
     * Delete service Group
     * @param $id
     * @return JsonResponse
     * @response 404 scenario="There is no service group with the specified ID"
     *  {
     *  "status": "error",
     *  "message": "There is no service group with this ID",
     *  "data": ""
     *  }
     * @response 200 scenario="the service group has been deleted successfully"
     * {
     * "status": "Request was successful",
     * "message": "the service group has been  deleted successfully",
     * "data": null
     * }
     */
    public function destroy($id): JsonResponse
    {
        try {
            $serviceGroup = ServiceGroup::findOrFail($id);
            $serviceGroup->delete();
            return $this->success(null,"the service group has been  deleted successfully");

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no service group with this ID", 404);
        }
    }
}
