<?php

namespace App\Http\Controllers\protobuf;

use App\Http\Requests\StorePatientRequest;
use App\Http\Resources\PatientCollection;
use App\Http\Resources\PatientResource;
use App\Models\Patient;
use App\Protos\PatientBroto as protoPatient;
use App\Traits\HttpResponses;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\Controller; // Added this import


class ProtoBufPatientController extends Controller
{
    use HttpResponses;

    public function index()
    {
        $patients = Patient::all();
        $patientProtos = [];

        foreach ($patients as $patient) {
            $patientProto = new protoPatient();
            $this->setPatientProtoData($patientProto, $patient);
            $patientProtos[] = $patientProto->serializeToJsonString();
        }

        return response()->json($patientProtos, 200);
    }

    public function store(StorePatientRequest $request): JsonResponse
    {
        $validatedData = $request->validated();
        $patientProto = new protoPatient();

        // Populate Proto from request data
        $patientProto->mergeFromJsonString(json_encode($validatedData));

        // Convert Proto back to array for database storage
        $patientData = $this->getPatientDataFromProto($patientProto);

        $patient = Patient::create($patientData);

        // Create response Proto
        $responseProto = new protoPatient();
        $this->setPatientProtoData($responseProto, $patient);

        return response()->json(
            json_decode($responseProto->serializeToJsonString()),
            200
        );
    }

    public function showById($id): JsonResponse
    {
        try {
            $patient = Patient::findOrFail($id);
            $patientProto = new protoPatient();
            $this->setPatientProtoData($patientProto, $patient);

            return response()->json(
                json_decode($patientProto->serializeToJsonString()),
                200
            );
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'No patient found with this ID'
            ], 404);
        }
    }

    public function show(Request $request): JsonResponse
    {
        $searchTerm = $request->input('search');
        $option = $request->input('option');

        if (empty($searchTerm)) {
            return response()->json([
                'status' => 'error',
                'message' => 'Search term is required'
            ], 400);
        }

        if (!in_array($option, ['all', 'name', 'file_number', 'civil_id', 'phone'])) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid option provided'
            ], 400);
        }

        try {
            $query = Patient::query();
            $this->buildSearchQuery($query, $option, $searchTerm);
            $patients = $query->get();

            if ($patients->isEmpty()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'No patients found matching the criteria'
                ], 404);
            }

            $patientProtos = [];
            foreach ($patients as $patient) {
                $patientProto = new protoPatient();
                $this->setPatientProtoData($patientProto, $patient);
                $patientProtos[] = json_decode($patientProto->serializeToJsonString());
            }

            return response()->json($patientProtos, 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred during search'
            ], 500);
        }
    }

    public function update(StorePatientRequest $request, $id): JsonResponse
    {
        try {
            $patient = Patient::findOrFail($id);
            $validatedData = $request->validated();

            $patientProto = new protoPatient();
            $patientProto->mergeFromJsonString(json_encode($validatedData));

            $patientData = $this->getPatientDataFromProto($patientProto);
            $patient->update($patientData);

            // Create response Proto
            $responseProto = new protoPatient();
            $this->setPatientProtoData($responseProto, $patient);

            return response()->json(
                json_decode($responseProto->serializeToJsonString()),
                200
            );
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'There is no patient with this ID'
            ], 404);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            $patient = Patient::findOrFail($id);
            $patient->delete();

            return response()->json([
                'status' => 'success',
                'message' => 'Patient deleted successfully'
            ], 200);
        } catch (ModelNotFoundException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'There is no patient with this ID'
            ], 404);
        }
    }

    private function setPatientProtoData(protoPatient $proto, Patient $patient): void
    {
        $proto->setId($patient->id);
        $proto->setNameEnglish($patient->name_english);
        $proto->setNameArabic($patient->name_arabic);
        $proto->setEmail($patient->email);
        $proto->setCivilId($patient->civil_id);
        $proto->setPhoneNumber($patient->phone_number);
        $proto->setPhoneHome($patient->phone_home);
        $proto->setNationality($patient->nationality);
        $proto->setCity($patient->city);
        $proto->setGender($patient->gender);
        $proto->setBirthDate($patient->birth_date);
        $proto->setAddress($patient->address);
        $proto->setInsuranceCompany($patient->insurance_company);
        $proto->setPolicyHolder($patient->policy_holder);
        $proto->setPolicyNumber($patient->policy_number);
        $proto->setPolicyStartDate($patient->policy_start_date);
        $proto->setPolicyEndDate($patient->policy_end_date);
        $proto->setNotice($patient->notice);
        $proto->setReferral($patient->referral);
    }

    private function getPatientDataFromProto(protoPatient $proto): array
    {
        return [
            'id' => $proto->getId(),
            'name_english' => $proto->getNameEnglish(),
            'name_arabic' => $proto->getNameArabic(),
            'email' => $proto->getEmail(),
            'civil_id' => $proto->getCivilId(),
            'phone_number' => $proto->getPhoneNumber(),
            'phone_home' => $proto->getPhoneHome(),
            'nationality' => $proto->getNationality(),
            'city' => $proto->getCity(),
            'gender' => $proto->getGender(),
            'birth_date' => $proto->getBirthDate(),
            'address' => $proto->getAddress(),
            'insurance_company' => $proto->getInsuranceCompany(),
            'policy_holder' => $proto->getPolicyHolder(),
            'policy_number' => $proto->getPolicyNumber(),
            'policy_start_date' => $proto->getPolicyStartDate(),
            'policy_end_date' => $proto->getPolicyEndDate(),
            'notice' => $proto->getNotice(),
            'referral' => $proto->getReferral(),
        ];
    }

    private function buildSearchQuery($query, $option, $searchTerm): void
    {
        if ($option == 'all') {
            $query->where(function ($q) use ($searchTerm) {
                $q->where('id', $searchTerm)
                    ->orWhere('name_english', 'LIKE', '%' . $searchTerm . '%')
                    ->orWhere('name_arabic', 'LIKE', '%' . $searchTerm . '%')
                    ->orWhere('civil_id', 'LIKE', '%' . $searchTerm . '%')
                    ->orWhere('phone_number', 'LIKE', '%' . $searchTerm . '%')
                    ->orWhere('phone_home', 'LIKE', '%' . $searchTerm . '%');
            });
        } elseif ($option == 'file_number') {
            $query->where('id', $searchTerm);
        } elseif ($option == 'name') {
            $query->where('name_english', 'LIKE', '%' . $searchTerm . '%')
                ->orWhere('name_arabic', 'LIKE', '%' . $searchTerm . '%');
        } elseif ($option == 'civil_id') {
            $query->where('civil_id', 'LIKE', '%' . $searchTerm . '%');
        } elseif ($option == 'phone') {
            $query->where('phone_number', 'LIKE', '%' . $searchTerm . '%')
                ->orWhere('phone_home', 'LIKE', '%' . $searchTerm . '%');
        }
    }
}
