<?php
namespace App\Http\Controllers;

use App\Http\Requests\GetAppointmentRequest;
use App\Http\Requests\StoreAppointmentRequest;
use App\Models\Appointment;
use App\Models\Doctor;
use App\Models\Service;
use App\Traits\HttpResponses;
use Illuminate\Http\JsonResponse;

class AppointmentController extends Controller
{
    use HttpResponses;
    /**
     * Store a newly created resource in storage.
     */
//    public function store(StoreAppointmentRequest $request): JsonResponse
//    {
//        // Retrieve the validated data with IDs
//        $validated = $request->validated();
//
//        // Find the doctor and service by name to get their IDs
//        $doctor = Doctor::where('name', $validated['doctor_name'])->firstOrFail();
//        $service = Service::where('name_english', $validated['service_name'])->firstOrFail();
//
//        // Create the appointment with the 'reserved' tag by default
//        $appointment = Appointment::create([
//            'doctor_id' => $doctor->id,
//            'service_id' => $service->id,
//            'date' => $validated['date'],
//            'start_time' => $validated['start_time'],
//            'end_time' => $validated['end_time'],
//            'tag' => 'reserved',
//        ]);
//
//        // Generate intervals for the given doctor and date
//        $intervals = $this->generateIntervals($doctor->id, $validated['date']);
//
//        return response()->json([
//            'status' => 'success',
//            'message' => 'Appointment created successfully.',
//            'appointment' => $appointment,
//            'intervals' => $intervals,
//        ]);
//    }


    public function store(StoreAppointmentRequest $request): JsonResponse
    {
        // Retrieve the validated data with IDs
        $validated = $request->validated();

        // Find the doctor and service by name to get their IDs
        $doctor = Doctor::where('name', $validated['doctor_name'])->firstOrFail();
        $service = Service::where('name_english', $validated['service_name'])->firstOrFail();

        // Check if the appointment time is within the doctor's available time
        if ($validated['start_time'] < $doctor->start_time || $validated['end_time'] > $doctor->end_time) {
            return response()->json([
                'status' => 'error',
                'message' => 'Appointment time is outside of the doctor\'s available hours.',
            ], 422);
        }

        // Create the appointment with the 'reserved' tag by default
        $appointment = Appointment::create([
            'doctor_id' => $doctor->id,
            'service_id' => $service->id,
            'date' => $validated['date'],
            'start_time' => $validated['start_time'],
            'end_time' => $validated['end_time'],
            'tag' => 'reserved',
        ]);

        // Generate intervals for the given doctor and date
        $intervals = $this->generateIntervals($doctor->id, $validated['date']);

        return response()->json([
            'status' => 'success',
            'message' => 'Appointment created successfully.',
            'appointment' => $appointment,
            'intervals' => $intervals,
        ]);
    }



    public function update(StoreAppointmentRequest $request , $id): JsonResponse
    {

        $appointment = Appointment::findOrFail($id);
        $validated = $request->validated();

        // Find the doctor and service by name to get their IDs
        $doctor = Doctor::where('name', $validated['doctor_name'])->firstOrFail();
        $service = Service::where('name_english', $validated['service_name'])->firstOrFail();

        // Create the appointment with the 'reserved' tag by default
        $appointmentData=$appointment->updateOrCreate([
            'doctor_id' => $doctor->id,
            'service_id' => $service->id,
            'date' => $validated['date'],
            'start_time' => $validated['start_time'],
            'end_time' => $validated['end_time'],
            'tag' => 'reserved',
        ]);


        return response()->json([
            'status' => 'success',
            'message' => 'Appointment has been updated successfully.',
            'appointment' => $appointmentData,
        ]);


    }
        public function getAppointments(GetAppointmentRequest $request): JsonResponse
    {
        // Validate only specific fields
        $validatedData = $request->validated();

        // Retrieve the validated doctor name and date
        $doctorName = $validatedData['doctor_name'];
        $date = $validatedData['date'];

        // Find the doctor by name
        $doctor = Doctor::where('name', $doctorName)->firstOrFail();

        // Find appointments for the doctor on the given date
        $appointments = Appointment::where('doctor_id', $doctor->id)
            ->where('date', $date)
            ->get();

        // Generate intervals for the given doctor and date
        $intervals = $this->generateIntervals($doctor->id, $date);

        return response()->json([
            'status' => 'success',
            'message' => 'Appointments retrieved successfully.',
            'appointments' => $appointments,
            'intervals' => $intervals,
        ]);
    }


//    public function getAppointmentsByDoctor($doctor_name)
//    {
//
//        $appointment=Appointment::all();;
//        $doctor = Doctor::where('name',$doctor_name)->firstOrFail();
//        $intervals = $this->generateIntervals($doctor->id, 'date');
//        return response()->json([
//            'appointment' => $appointment,
//            'intervals' => $intervals,
//        ]);
//    }

    /**
     * Get appointments and intervals for a specific doctor.
     */


//    public function getAppointmentsByDoctor($doctorName): JsonResponse
//    {
//        // Find the doctor by name
//        $doctor = Doctor::where('name', $doctorName)->firstOrFail();
//
//        // Retrieve all appointments for the given doctor
//        $appointments = Appointment::where('doctor_id', $doctor->id)
//            ->orderBy('date')
//            ->orderBy('start_time')
//            ->get();
//
//        // Generate intervals for the given doctor and date
//        $date = now()->format('Y-m-d'); // Or use a specific date if needed
//        $intervals = $this->generateIntervals($doctor->id, $date);
//
//        return response()->json([
//            'status' => 'success',
//            'appointments' => $appointments,
//            'intervals' => $intervals,
//        ]);
//    }

    public function getAppointmentsByDoctor($doctorName): JsonResponse
    {
        // Find the doctor by name
//        dd($doctorName);
        $doctor = Doctor::where('name', $doctorName)->first();
//        if ($doctor) {
//            dd($doctor->name);
//        } else {
//            // Handle the case where no doctor was found
//            dd('Doctor not found');
//        }        // Retrieve all appointments for the given doctor
        $appointments = Appointment::where('doctor_id', $doctor->id)
            ->orderBy('date')
            ->orderBy('start_time')
            ->get();

//        // Use the provided date or default to today
//        $date = $date ?? now()->format('Y-m-d');

        // Generate intervals for the given doctor and date
        $intervals = $this->generateIntervals($doctor->id,$date=null);

        return response()->json([
            'status' => 'success',
            'appointments' => $appointments,
            'intervals' => $intervals,
        ]);
    }

protected function destroy($id):JsonResponse{
    $appointment = Appointment::findOrFail($id);
    $appointment->delete();
    return $this->success(null,"the patient appointment has been deleted successfully");
}

    /**
     * Generate intervals for a given doctor and date.
     */
    protected function generateIntervals($doctorId, $date): array
    {
        $doctor = Doctor::where('id', $doctorId)->firstOrFail();
        // Fetch all appointments for the given doctor and date
        $appointments = Appointment::where('doctor_id', $doctorId)
            ->where('date', $date)
            ->orderBy('start_time')
            ->get();

        $startOfDay = $doctor->start_time;
        $endOfDay = $doctor->end_time;

        // Initialize intervals array with the whole day as 'not reserved'
        $intervals = [
            [
                'start_time' => $startOfDay,
                'end_time' => $endOfDay,
                'tag' => 'not reserved',
            ]
        ];

        // Process each appointment to adjust intervals
        foreach ($appointments as $appointment) {
            $newIntervals = [];
            foreach ($intervals as $interval) {
                // If appointment starts before the interval ends and ends after the interval starts
                if ($appointment->start_time < $interval['end_time'] && $appointment->end_time > $interval['start_time']) {
                    // Split before the appointment if necessary
                    if ($appointment->start_time > $interval['start_time']) {
                        $newIntervals[] = [
                            'start_time' => $interval['start_time'],
                            'end_time' => $appointment->start_time,
                            'tag' => 'not reserved',
                        ];
                    }

                    // Mark the appointment time as 'reserved'
                    $newIntervals[] = [
                        'start_time' => max($interval['start_time'], $appointment->start_time),
                        'end_time' => min($interval['end_time'], $appointment->end_time),
                        'tag' => 'reserved',
                    ];

                    // Split after the appointment if necessary
                    if ($appointment->end_time < $interval['end_time']) {
                        $newIntervals[] = [
                            'start_time' => $appointment->end_time,
                            'end_time' => $interval['end_time'],
                            'tag' => 'not reserved',
                        ];
                    }
                } else {
                    // If the interval is completely outside of the appointment time
                    $newIntervals[] = $interval;
                }
            }
            $intervals = $newIntervals;
        }

        // Ensure no duplicate intervals
        $intervals = array_unique($intervals, SORT_REGULAR);

        return $intervals;
    }
}
