<?php
namespace App\Http\Controllers;

use App\Http\Requests\LoginUserRequest;
use App\Http\Requests\ResetPasswordRequest;
use App\Http\Requests\StoreUserRequest;
use App\Models\User;
use App\Traits\HttpResponses;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
/**
 * @group Auth management
 *
 * APIs to manage Auth resources
 * */
class AuthController extends Controller
{
    use HttpResponses;

    /**
     * login
     * @param LoginUserRequest $request
     * @return \Illuminate\Http\JsonResponse
     * @response 200 scenario="user logged in successfully"
     * {
     * "status": "Request was successful",
     * "message": "user logged in successfully",
     * "data": {
     * "user": {
     * "id": 1,
     * "name": "ahmed ismail",
     * "created_at": "2024-10-12T22:04:34.000000Z",
     * "updated_at": "2024-10-12T22:04:34.000000Z"
     * },
     * "token": "5|xK7B7IWh0cEdrEwTJCQJwGWUnsEvMTc6QxkAFFVz59a8a498"
     * }
     * }
     * @response 404 scenario="the specified user doesn't exist"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "name": [
     * "This user doesn't exist."
     * ]
     * }
     * }
     * @response 404 scenario="wrong password"
     * {
     * "status": "error",
     * "message": "the password isn't correct",
     * "data": null
     * }
     */
    public function login(LoginUserRequest $request): \Illuminate\Http\JsonResponse
    {
        $credentials = $request->only('name', 'password');

        if (!Auth::attempt($credentials)) {
            return $this->error(null, "the password isn't correct", 401);
        }
        $user = Auth::user();
        return $this->success([
            'user' => $user,
            'token' => $user->createToken('Token of ' . $user->name)->plainTextToken,
        ],"user logged in successfully");
    }

    /**
     * create a user
     * @param StoreUserRequest $request
     * @return \Illuminate\Http\JsonResponse
     * @response 200 scenario="user has been created successfully"
     * {
     * "status": "Request was successful",
     * "message": "user has been created successfully",
     * "data": {
     * "user": {
     * "name": "ahmed",
     * "updated_at": "2024-10-12T22:07:57.000000Z",
     * "created_at": "2024-10-12T22:07:57.000000Z",
     * "id": 3
     * },
     * "token": "3|vUhBeVm8bRTlKl8UsgInOL7p6RGvHiqeuxZYpngJ5e93c06e"
     * }
     * }
     * @response 404 scenario="username has already been taken"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "name": [
     * "This username has already been taken."
     * ]
     * }
     * }
     */
    public function register(StoreUserRequest $request): \Illuminate\Http\JsonResponse
    {
        $user = User::create([
            'name' => $request->name,
            'password' => Hash::make($request->password),
        ]);

        return $this->success([
            'user' => $user,
            'token' => $user->createToken('Token of ' . $user->name)->plainTextToken,
        ],"user has been created successfully");

    }

    /**
     * reset password
     * @param ResetPasswordRequest $request
     * @return JsonResponse
     * @response 200 scenario="password reset successfully"
     * {
     * "status": "Request was successful",
     * "message": "Password has been successfully updated.",
     * "data": null
     * }
 * }
     * @response 404 scenario="incorrect old password"
     * {
     * "status": "error",
     * "message": "The old password isn't correct.",
     * "data": null
     * }
     * @response 404 scenario="new password confirmation does not match"
     * {
     * "status": "error",
     * "message": "Validation failed",
     * "data": {
     * "new_password": [
     * "The new password confirmation does not match."
     * ]
     * }
     * }
     */
    public function resetPassword(ResetPasswordRequest $request): \Illuminate\Http\JsonResponse
    {
        $user = User::where('name', $request->name)->first();

        if (!Hash::check($request->old_password, $user->password)) {
            return $this->error(null, "The old password isn't correct.", 401);
        }

        $user->password = Hash::make($request->new_password);
        $user->save();

        return $this->success(null,'Password has been successfully updated.');
    }

    /**
     * logout
     * @return \Illuminate\Http\JsonResponse
     * @response 200 scenario="logged out successfully"
     * {
     * "status": "Request was successful",
     * "message": "logged out successfully",
     * "data": null
     * }
     * @response 404 scenario="unauthenticated"
     * {
     * "message": "Unauthenticated."
     * }
     *
    */
    public function logout(): JsonResponse
    {
        // Revoke the current user's token
        try {
        Auth::user()->currentAccessToken()->delete();
        }catch (\Exception $e){
            return $this->error(null,"this user either have no account or didn't login",404);
        }

        // Return a success response
        return $this->success(null,"logged out successfully");
    }
}
