<?php

namespace App\Http\Controllers;

use App\Http\Requests\StorePatientRequest;
use App\Http\Resources\PatientCollection;
use App\Http\Resources\PatientResource;
use App\Models\Patient;
use App\Traits\HttpResponses;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use function PHPUnit\Framework\returnArgument;

/**
 * @group Patient management
 *
 * APIs to manage Patient resources
 * */
class PatientController extends Controller
{
    use HttpResponses;
    /**
     * Display all patients
     *
     * @response 200 scenario="success"
     * {
     * "status": "Request was successful",
     * "message": "success",
     * "data": [
     * {
     * "id": 2,
     * "name_english": "ali mohamed",
     * "name_arabic": "أحمد علي",
     * "email": "ahmed@gmail.com",
     * "civil_id": "30201078908135",
     * "phone_number": "01145684288",
     * "nationality": "egyption",
     * "city": "menofia",
     * "gender": "male",
     * "birth_date": "2002-08-08",
     * "address": "64 altowheed stread",
     * "insurance_company": "bank masr for life ensurance",
     * "policy_holder": "mohomed hatem",
     * "policy_number": "2262509",
     * "policy_start_date": "2006-08-08",
     * "policy_end_date": "2025-08-09",
     * "notice": "i noticed",
     * "how_did_you_know_about_us": "internet",
     * "patient_status": null,
     * "report_1": null,
     * "report_2": null,
     * "other_document": null,
     * "transactions": []
     * }
     * ]
     * }
     */
    public function index(): JsonResponse
    {
        $patient = Patient::all();
        return $this->success(new PatientCollection($patient),"success");
    }


    /*
     * store new patient
     * @bodyParam name_english string required The patient's name in English. Example: App\Patient::factory()->make()->name_english
     * @bodyParam name_arabic string required The patient's name in Arabic. Example: App\Patient::factory()->make()->name_arabic
     * @bodyParam civil_id string required The patient's 14-digit civil ID. Example: App\Patient::factory()->make()->civil_id
     * @bodyParam phone_number string required The patient's phone number. Example: App\Patient::factory()->make()->phone_number
     * @bodyParam nationality string required The patient's nationality. Example: App\Patient::factory()->make()->nationality
     * @bodyParam city string required The city where the patient lives. Example: App\Patient::factory()->make()->city
     * @bodyParam gender string required The patient's gender. Must be 'male' or 'female'. Example: App\Patient::factory()->make()->gender
     * @bodyParam birth_date date required The patient's birth_date in 'YYYY-MM-DD' format. Example: App\Patient::factory()->make()->birth_date
     * @bodyParam address string optional The patient's address. Example: App\Patient::factory()->make()->address
     * @bodyParam insurance_company string optional The insurance company name. Example: App\Patient::factory()->make()->insurance_company
     * @bodyParam policy_holder string optional The policy_holder name. Example: App\Patient::factory()->make()->policy_holder
     * @bodyParam policy_number string optional The policy number. Example: App\Patient::factory()->make()->policy_number
     * @bodyParam policy_start_date date optional The policy start date in 'YYYY-MM-DD' format. Example: App\Patient::factory()->make()->policy_start_date
     * @bodyParam policy_end_date date optional The policy end date in 'YYYY-MM-DD' format. Example: App\Patient::factory()->make()->policy_end_date
     */

     /**
      *Create patient
      * @response 200 scenario="patient has been created successfully"
      * {
      * "status": "Request was successful",
      * "message": "patient has been created successfully",
      * "data": {
      * "id": 2,
      * "name_english": "ali mohamed",
      * "name_arabic": "أحمد علي",
      * "email": "ahmed@gmail.com",
      * "civil_id": "30201078908135",
      * "phone_number": "01145684288",
      * "nationality": "egyption",
      * "city": "menofia",
      * "gender": "male",
      * "birth_date": "2002-8-8",
      * "address": "64 altowheed stread",
      * "insurance_company": "bank masr for life ensurance",
      * "policy_holder": "mohomed hatem",
      * "policy_number": "2262509",
      * "policy_start_date": "2006-8-8",
      * "policy_end_date": "2025-8-9",
      * "notice": "i noticed",
      * "how_did_you_know_about_us": "internet",
      * "patient_status": null,
      * "report_1": null,
      * "report_2": null,
      * "other_document": null,
      * "transactions": []
      * }
      * }
      * @response 404 scenario="civil id or email already exist"
      * {
      * "status": "error",
      * "message": "Validation failed",
      * "data": {
      * "civil_id": [
      * "This civil id already exists."
      * ],
      * "email": [
      * "The email has already been taken."
      * ]
      * }
      * }
      * @bodyParam name_english string required The patient's name in English. Example: <faker>name</faker>
      * @bodyParam name_arabic string required The patient's name in Arabic. Example: <faker>name</faker>
      * @bodyParam civil_id string required The patient's 14-digit civil ID. Example: <faker>numerify('##############')</faker>
      * @bodyParam phone_number string required The patient's phone number. Example: <faker>e164PhoneNumber</faker>
      * @bodyParam nationality string required The patient's nationality. Example: <faker>country</faker>
      * @bodyParam city string required The city where the patient lives. Example: <faker>city</faker>
      * @bodyParam gender string required The patient's gender. Must be 'male' or 'female'. Example: <faker>randomElement(['male', 'female'])</faker>
      * @bodyParam birth_date date required The patient's birth date in 'YYYY-MM-DD' format. Example: <faker>date('Y-m-d')</faker>
      * @bodyParam address string optional The patient's address. Example: <faker>address</faker>
      * @bodyParam insurance_company string optional The insurance company name. Example: <faker>company</faker>
      * @bodyParam policy_holder string optional The policy holder's name. Example: <faker>name</faker>
      * @bodyParam policy_number string optional The policy number. Example: <faker>uuid</faker>
      * @bodyParam policy_start_date date optional The policy start date in 'YYYY-MM-DD' format. Example: <faker>date('Y-m-d')</faker>
      * @bodyParam policy_end_date date optional The policy end date in 'YYYY-MM-DD' format. Example: <faker>date('Y-m-d')</faker>
      */
    public function store(StorePatientRequest $request): JsonResponse
    {
        $validatedData = $request->validated();
        $patient = Patient::create($validatedData);
//        return $this->success(
//            ['patient'=> $patient]
//        );
        return $this->success(new PatientResource($patient),"patient has been created successfully");
    }

    /**
     * Display patient by id
     * @response 200 scenario="request was successful"
     * {
     * "status": "Request was successful",
     * "message": "success",
     * "data": {
     * "id": 2,
     * "name_english": "ali mohamed",
     * "name_arabic": "أحمد علي",
     * "email": "ahmed@gmail.com",
     * "civil_id": "30201078908135",
     * "phone_number": "01145684288",
     * "nationality": "egyption",
     * "city": "menofia",
     * "gender": "male",
     * "birth_date": "2002-08-08",
     * "address": "64 altowheed stread",
     * "insurance_company": "bank masr for life ensurance",
     * "policy_holder": "mohomed hatem",
     * "policy_number": "2262509",
     * "policy_start_date": "2006-08-08",
     * "policy_end_date": "2025-08-09",
     * "notice": "i noticed",
     * "how_did_you_know_about_us": "internet",
     * "patient_status": null,
     * "report_1": null,
     * "report_2": null,
     * "other_document": null,
     * "transactions": []
     * }
     * }
     * @response 404 scenario="no patient with the specified ID"
     * {
     * "status": "error",
     * "message": "There is no patient with this ID",
     * "data": ""
     * }
     */
    public function show($id): JsonResponse
    {
        try {
            $patient = Patient::findOrFail($id);
            return $this->success(new PatientResource($patient),"success");

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no patient with this ID", 404);
        }
    }

    /**
     * Update patient.
     * @response 404 scenario="no patient with the specified ID"
     *  {
     *  "status": "error",
     *  "message": "There is no patient with this ID",
     *  "data": ""
     *  }
     * @response 200 scenario="patient has been updated successfully"
     * {
     * "status": "Request was successful",
     * "message": "the patient was updated successfully",
     * "data": {
     * "id": 3,
     * "name_english": "abdelfatah mahmoud housein",
     * "name_arabic": "أحمد علي",
     * "email": "ahmed@gmail.com",
     * "civil_id": "30204532890541",
     * "phone_number": "01145684288",
     * "nationality": "egyption",
     * "city": "menofia",
     * "gender": "male",
     * "birth_date": "2002-8-20",
     * "address": "64 altowheed stread",
     * "insurance_company": "bank masr for life ensurance",
     * "policy_holder": "mohomed hatem",
     * "policy_number": "2262509",
     * "policy_start_date": "2006-8-8",
     * "policy_end_date": "2025-8-9",
     * "notice": "i noticed",
     * "how_did_you_know_about_us": "internet",
     * "patient_status": null,
     * "report_1": null,
     * "report_2": null,
     * "other_document": null,
     * "transactions": []
     * }
     * }
     */
    public function update(StorePatientRequest $request, $id):JsonResponse
    {
        try {
            $patient = Patient::findOrFail($id);
            $validatedData = $request->validated();
            $patient->update($validatedData);
            return $this->success(new PatientResource($patient), "the patient was updated successfully");
        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no patient with this ID", 404);
        }
    }
    /**
     * Delete patient.
     * @response 404 scenario="no patient with the specified ID"
     *   {
     *   "status": "error",
     *   "message": "There is no patient with this ID",
     *   "data": ""
     *   }
     *
     * @response 200 scenario="patient has been deleted successfully"
     * {
     * "status": "Request was successful",
     * "message": "the patient has been  deleted successfully",
     * "data": null
     * }
     */
    public function destroy($id): JsonResponse
    {
        try {
            $patient = Patient::findOrFail($id);
            $patient->delete();
            return $this->success(null,"the patient has been  deleted successfully");

        } catch (ModelNotFoundException $e) {
            return $this->error("", "There is no patient with this ID", 404);
        }
    }
}
